// Copyright (c) 2025 Karl Gaissmaier
// SPDX-License-Identifier: MIT

// Package lpm (longest-prefix-match) contains the lookup table with which
// the backtracking for the lpm in the complete binary tree of the prefixes
// can be replaced by a fast bitset operation.
package lpm

import "github.com/gaissmai/bart/internal/bitset"

// BackTrackingBitset is the backtracking sequence in the complete binary tree
// of the prefixes (mapped by the base_index function) as bitstring.
//
//	for idx := 1; idx > 0; idx >>= 1 { b.Set(idx) }
//
// allows a one shot bitset intersection algorithm:
//
//	func (n *node[V]) lpmTest(idx uint) bool {
//		return n.prefixes.IntersectsAny(lpmbt.LookupTbl[idx])
//	}
//
// instead of a sequence of single bitset tests:
//
//	func (n *node[V]) lpmTest(idx uint) bool {
//		for ; idx > 0; idx >>= 1 {
//			if n.prefixes.Test(idx) {
//				return true
//			}
//		}
//		return false
//	}
func BackTrackingBitset(idx uint) *bitset.BitSet256 {
	return &lookupTbl[idx&511] // &511 is BCE
}

var lookupTbl = [512]bitset.BitSet256{
	/* idx:   0 */ {0x0, 0x0, 0x0, 0x0}, // invalid
	/* idx:   1 */ {0x2, 0x0, 0x0, 0x0}, // [1]
	/* idx:   2 */ {0x6, 0x0, 0x0, 0x0}, // [1 2]
	/* idx:   3 */ {0xa, 0x0, 0x0, 0x0}, // [1 3]
	/* idx:   4 */ {0x16, 0x0, 0x0, 0x0}, // [1 2 4]
	/* idx:   5 */ {0x26, 0x0, 0x0, 0x0}, // [1 2 5]
	/* idx:   6 */ {0x4a, 0x0, 0x0, 0x0}, // [1 3 6]
	/* idx:   7 */ {0x8a, 0x0, 0x0, 0x0}, // [1 3 7]
	/* idx:   8 */ {0x116, 0x0, 0x0, 0x0}, // [1 2 4 8]
	/* idx:   9 */ {0x216, 0x0, 0x0, 0x0}, // [1 2 4 9]
	/* idx:  10 */ {0x426, 0x0, 0x0, 0x0}, // [1 2 5 10]
	/* idx:  11 */ {0x826, 0x0, 0x0, 0x0}, // [1 2 5 11]
	/* idx:  12 */ {0x104a, 0x0, 0x0, 0x0}, // [1 3 6 12]
	/* idx:  13 */ {0x204a, 0x0, 0x0, 0x0}, // [1 3 6 13]
	/* idx:  14 */ {0x408a, 0x0, 0x0, 0x0}, // [1 3 7 14]
	/* idx:  15 */ {0x808a, 0x0, 0x0, 0x0}, // [1 3 7 15]
	/* idx:  16 */ {0x10116, 0x0, 0x0, 0x0}, // [1 2 4 8 16]
	/* idx:  17 */ {0x20116, 0x0, 0x0, 0x0}, // [1 2 4 8 17]
	/* idx:  18 */ {0x40216, 0x0, 0x0, 0x0}, // [1 2 4 9 18]
	/* idx:  19 */ {0x80216, 0x0, 0x0, 0x0}, // [1 2 4 9 19]
	/* idx:  20 */ {0x100426, 0x0, 0x0, 0x0}, // [1 2 5 10 20]
	/* idx:  21 */ {0x200426, 0x0, 0x0, 0x0}, // [1 2 5 10 21]
	/* idx:  22 */ {0x400826, 0x0, 0x0, 0x0}, // [1 2 5 11 22]
	/* idx:  23 */ {0x800826, 0x0, 0x0, 0x0}, // [1 2 5 11 23]
	/* idx:  24 */ {0x100104a, 0x0, 0x0, 0x0}, // [1 3 6 12 24]
	/* idx:  25 */ {0x200104a, 0x0, 0x0, 0x0}, // [1 3 6 12 25]
	/* idx:  26 */ {0x400204a, 0x0, 0x0, 0x0}, // [1 3 6 13 26]
	/* idx:  27 */ {0x800204a, 0x0, 0x0, 0x0}, // [1 3 6 13 27]
	/* idx:  28 */ {0x1000408a, 0x0, 0x0, 0x0}, // [1 3 7 14 28]
	/* idx:  29 */ {0x2000408a, 0x0, 0x0, 0x0}, // [1 3 7 14 29]
	/* idx:  30 */ {0x4000808a, 0x0, 0x0, 0x0}, // [1 3 7 15 30]
	/* idx:  31 */ {0x8000808a, 0x0, 0x0, 0x0}, // [1 3 7 15 31]
	/* idx:  32 */ {0x100010116, 0x0, 0x0, 0x0}, // [1 2 4 8 16 32]
	/* idx:  33 */ {0x200010116, 0x0, 0x0, 0x0}, // [1 2 4 8 16 33]
	/* idx:  34 */ {0x400020116, 0x0, 0x0, 0x0}, // [1 2 4 8 17 34]
	/* idx:  35 */ {0x800020116, 0x0, 0x0, 0x0}, // [1 2 4 8 17 35]
	/* idx:  36 */ {0x1000040216, 0x0, 0x0, 0x0}, // [1 2 4 9 18 36]
	/* idx:  37 */ {0x2000040216, 0x0, 0x0, 0x0}, // [1 2 4 9 18 37]
	/* idx:  38 */ {0x4000080216, 0x0, 0x0, 0x0}, // [1 2 4 9 19 38]
	/* idx:  39 */ {0x8000080216, 0x0, 0x0, 0x0}, // [1 2 4 9 19 39]
	/* idx:  40 */ {0x10000100426, 0x0, 0x0, 0x0}, // [1 2 5 10 20 40]
	/* idx:  41 */ {0x20000100426, 0x0, 0x0, 0x0}, // [1 2 5 10 20 41]
	/* idx:  42 */ {0x40000200426, 0x0, 0x0, 0x0}, // [1 2 5 10 21 42]
	/* idx:  43 */ {0x80000200426, 0x0, 0x0, 0x0}, // [1 2 5 10 21 43]
	/* idx:  44 */ {0x100000400826, 0x0, 0x0, 0x0}, // [1 2 5 11 22 44]
	/* idx:  45 */ {0x200000400826, 0x0, 0x0, 0x0}, // [1 2 5 11 22 45]
	/* idx:  46 */ {0x400000800826, 0x0, 0x0, 0x0}, // [1 2 5 11 23 46]
	/* idx:  47 */ {0x800000800826, 0x0, 0x0, 0x0}, // [1 2 5 11 23 47]
	/* idx:  48 */ {0x100000100104a, 0x0, 0x0, 0x0}, // [1 3 6 12 24 48]
	/* idx:  49 */ {0x200000100104a, 0x0, 0x0, 0x0}, // [1 3 6 12 24 49]
	/* idx:  50 */ {0x400000200104a, 0x0, 0x0, 0x0}, // [1 3 6 12 25 50]
	/* idx:  51 */ {0x800000200104a, 0x0, 0x0, 0x0}, // [1 3 6 12 25 51]
	/* idx:  52 */ {0x1000000400204a, 0x0, 0x0, 0x0}, // [1 3 6 13 26 52]
	/* idx:  53 */ {0x2000000400204a, 0x0, 0x0, 0x0}, // [1 3 6 13 26 53]
	/* idx:  54 */ {0x4000000800204a, 0x0, 0x0, 0x0}, // [1 3 6 13 27 54]
	/* idx:  55 */ {0x8000000800204a, 0x0, 0x0, 0x0}, // [1 3 6 13 27 55]
	/* idx:  56 */ {0x10000001000408a, 0x0, 0x0, 0x0}, // [1 3 7 14 28 56]
	/* idx:  57 */ {0x20000001000408a, 0x0, 0x0, 0x0}, // [1 3 7 14 28 57]
	/* idx:  58 */ {0x40000002000408a, 0x0, 0x0, 0x0}, // [1 3 7 14 29 58]
	/* idx:  59 */ {0x80000002000408a, 0x0, 0x0, 0x0}, // [1 3 7 14 29 59]
	/* idx:  60 */ {0x100000004000808a, 0x0, 0x0, 0x0}, // [1 3 7 15 30 60]
	/* idx:  61 */ {0x200000004000808a, 0x0, 0x0, 0x0}, // [1 3 7 15 30 61]
	/* idx:  62 */ {0x400000008000808a, 0x0, 0x0, 0x0}, // [1 3 7 15 31 62]
	/* idx:  63 */ {0x800000008000808a, 0x0, 0x0, 0x0}, // [1 3 7 15 31 63]
	/* idx:  64 */ {0x100010116, 0x1, 0x0, 0x0}, // [1 2 4 8 16 32 64]
	/* idx:  65 */ {0x100010116, 0x2, 0x0, 0x0}, // [1 2 4 8 16 32 65]
	/* idx:  66 */ {0x200010116, 0x4, 0x0, 0x0}, // [1 2 4 8 16 33 66]
	/* idx:  67 */ {0x200010116, 0x8, 0x0, 0x0}, // [1 2 4 8 16 33 67]
	/* idx:  68 */ {0x400020116, 0x10, 0x0, 0x0}, // [1 2 4 8 17 34 68]
	/* idx:  69 */ {0x400020116, 0x20, 0x0, 0x0}, // [1 2 4 8 17 34 69]
	/* idx:  70 */ {0x800020116, 0x40, 0x0, 0x0}, // [1 2 4 8 17 35 70]
	/* idx:  71 */ {0x800020116, 0x80, 0x0, 0x0}, // [1 2 4 8 17 35 71]
	/* idx:  72 */ {0x1000040216, 0x100, 0x0, 0x0}, // [1 2 4 9 18 36 72]
	/* idx:  73 */ {0x1000040216, 0x200, 0x0, 0x0}, // [1 2 4 9 18 36 73]
	/* idx:  74 */ {0x2000040216, 0x400, 0x0, 0x0}, // [1 2 4 9 18 37 74]
	/* idx:  75 */ {0x2000040216, 0x800, 0x0, 0x0}, // [1 2 4 9 18 37 75]
	/* idx:  76 */ {0x4000080216, 0x1000, 0x0, 0x0}, // [1 2 4 9 19 38 76]
	/* idx:  77 */ {0x4000080216, 0x2000, 0x0, 0x0}, // [1 2 4 9 19 38 77]
	/* idx:  78 */ {0x8000080216, 0x4000, 0x0, 0x0}, // [1 2 4 9 19 39 78]
	/* idx:  79 */ {0x8000080216, 0x8000, 0x0, 0x0}, // [1 2 4 9 19 39 79]
	/* idx:  80 */ {0x10000100426, 0x10000, 0x0, 0x0}, // [1 2 5 10 20 40 80]
	/* idx:  81 */ {0x10000100426, 0x20000, 0x0, 0x0}, // [1 2 5 10 20 40 81]
	/* idx:  82 */ {0x20000100426, 0x40000, 0x0, 0x0}, // [1 2 5 10 20 41 82]
	/* idx:  83 */ {0x20000100426, 0x80000, 0x0, 0x0}, // [1 2 5 10 20 41 83]
	/* idx:  84 */ {0x40000200426, 0x100000, 0x0, 0x0}, // [1 2 5 10 21 42 84]
	/* idx:  85 */ {0x40000200426, 0x200000, 0x0, 0x0}, // [1 2 5 10 21 42 85]
	/* idx:  86 */ {0x80000200426, 0x400000, 0x0, 0x0}, // [1 2 5 10 21 43 86]
	/* idx:  87 */ {0x80000200426, 0x800000, 0x0, 0x0}, // [1 2 5 10 21 43 87]
	/* idx:  88 */ {0x100000400826, 0x1000000, 0x0, 0x0}, // [1 2 5 11 22 44 88]
	/* idx:  89 */ {0x100000400826, 0x2000000, 0x0, 0x0}, // [1 2 5 11 22 44 89]
	/* idx:  90 */ {0x200000400826, 0x4000000, 0x0, 0x0}, // [1 2 5 11 22 45 90]
	/* idx:  91 */ {0x200000400826, 0x8000000, 0x0, 0x0}, // [1 2 5 11 22 45 91]
	/* idx:  92 */ {0x400000800826, 0x10000000, 0x0, 0x0}, // [1 2 5 11 23 46 92]
	/* idx:  93 */ {0x400000800826, 0x20000000, 0x0, 0x0}, // [1 2 5 11 23 46 93]
	/* idx:  94 */ {0x800000800826, 0x40000000, 0x0, 0x0}, // [1 2 5 11 23 47 94]
	/* idx:  95 */ {0x800000800826, 0x80000000, 0x0, 0x0}, // [1 2 5 11 23 47 95]
	/* idx:  96 */ {0x100000100104a, 0x100000000, 0x0, 0x0}, // [1 3 6 12 24 48 96]
	/* idx:  97 */ {0x100000100104a, 0x200000000, 0x0, 0x0}, // [1 3 6 12 24 48 97]
	/* idx:  98 */ {0x200000100104a, 0x400000000, 0x0, 0x0}, // [1 3 6 12 24 49 98]
	/* idx:  99 */ {0x200000100104a, 0x800000000, 0x0, 0x0}, // [1 3 6 12 24 49 99]
	/* idx: 100 */ {0x400000200104a, 0x1000000000, 0x0, 0x0}, // [1 3 6 12 25 50 100]
	/* idx: 101 */ {0x400000200104a, 0x2000000000, 0x0, 0x0}, // [1 3 6 12 25 50 101]
	/* idx: 102 */ {0x800000200104a, 0x4000000000, 0x0, 0x0}, // [1 3 6 12 25 51 102]
	/* idx: 103 */ {0x800000200104a, 0x8000000000, 0x0, 0x0}, // [1 3 6 12 25 51 103]
	/* idx: 104 */ {0x1000000400204a, 0x10000000000, 0x0, 0x0}, // [1 3 6 13 26 52 104]
	/* idx: 105 */ {0x1000000400204a, 0x20000000000, 0x0, 0x0}, // [1 3 6 13 26 52 105]
	/* idx: 106 */ {0x2000000400204a, 0x40000000000, 0x0, 0x0}, // [1 3 6 13 26 53 106]
	/* idx: 107 */ {0x2000000400204a, 0x80000000000, 0x0, 0x0}, // [1 3 6 13 26 53 107]
	/* idx: 108 */ {0x4000000800204a, 0x100000000000, 0x0, 0x0}, // [1 3 6 13 27 54 108]
	/* idx: 109 */ {0x4000000800204a, 0x200000000000, 0x0, 0x0}, // [1 3 6 13 27 54 109]
	/* idx: 110 */ {0x8000000800204a, 0x400000000000, 0x0, 0x0}, // [1 3 6 13 27 55 110]
	/* idx: 111 */ {0x8000000800204a, 0x800000000000, 0x0, 0x0}, // [1 3 6 13 27 55 111]
	/* idx: 112 */ {0x10000001000408a, 0x1000000000000, 0x0, 0x0}, // [1 3 7 14 28 56 112]
	/* idx: 113 */ {0x10000001000408a, 0x2000000000000, 0x0, 0x0}, // [1 3 7 14 28 56 113]
	/* idx: 114 */ {0x20000001000408a, 0x4000000000000, 0x0, 0x0}, // [1 3 7 14 28 57 114]
	/* idx: 115 */ {0x20000001000408a, 0x8000000000000, 0x0, 0x0}, // [1 3 7 14 28 57 115]
	/* idx: 116 */ {0x40000002000408a, 0x10000000000000, 0x0, 0x0}, // [1 3 7 14 29 58 116]
	/* idx: 117 */ {0x40000002000408a, 0x20000000000000, 0x0, 0x0}, // [1 3 7 14 29 58 117]
	/* idx: 118 */ {0x80000002000408a, 0x40000000000000, 0x0, 0x0}, // [1 3 7 14 29 59 118]
	/* idx: 119 */ {0x80000002000408a, 0x80000000000000, 0x0, 0x0}, // [1 3 7 14 29 59 119]
	/* idx: 120 */ {0x100000004000808a, 0x100000000000000, 0x0, 0x0}, // [1 3 7 15 30 60 120]
	/* idx: 121 */ {0x100000004000808a, 0x200000000000000, 0x0, 0x0}, // [1 3 7 15 30 60 121]
	/* idx: 122 */ {0x200000004000808a, 0x400000000000000, 0x0, 0x0}, // [1 3 7 15 30 61 122]
	/* idx: 123 */ {0x200000004000808a, 0x800000000000000, 0x0, 0x0}, // [1 3 7 15 30 61 123]
	/* idx: 124 */ {0x400000008000808a, 0x1000000000000000, 0x0, 0x0}, // [1 3 7 15 31 62 124]
	/* idx: 125 */ {0x400000008000808a, 0x2000000000000000, 0x0, 0x0}, // [1 3 7 15 31 62 125]
	/* idx: 126 */ {0x800000008000808a, 0x4000000000000000, 0x0, 0x0}, // [1 3 7 15 31 63 126]
	/* idx: 127 */ {0x800000008000808a, 0x8000000000000000, 0x0, 0x0}, // [1 3 7 15 31 63 127]
	/* idx: 128 */ {0x100010116, 0x1, 0x1, 0x0}, // [1 2 4 8 16 32 64 128]
	/* idx: 129 */ {0x100010116, 0x1, 0x2, 0x0}, // [1 2 4 8 16 32 64 129]
	/* idx: 130 */ {0x100010116, 0x2, 0x4, 0x0}, // [1 2 4 8 16 32 65 130]
	/* idx: 131 */ {0x100010116, 0x2, 0x8, 0x0}, // [1 2 4 8 16 32 65 131]
	/* idx: 132 */ {0x200010116, 0x4, 0x10, 0x0}, // [1 2 4 8 16 33 66 132]
	/* idx: 133 */ {0x200010116, 0x4, 0x20, 0x0}, // [1 2 4 8 16 33 66 133]
	/* idx: 134 */ {0x200010116, 0x8, 0x40, 0x0}, // [1 2 4 8 16 33 67 134]
	/* idx: 135 */ {0x200010116, 0x8, 0x80, 0x0}, // [1 2 4 8 16 33 67 135]
	/* idx: 136 */ {0x400020116, 0x10, 0x100, 0x0}, // [1 2 4 8 17 34 68 136]
	/* idx: 137 */ {0x400020116, 0x10, 0x200, 0x0}, // [1 2 4 8 17 34 68 137]
	/* idx: 138 */ {0x400020116, 0x20, 0x400, 0x0}, // [1 2 4 8 17 34 69 138]
	/* idx: 139 */ {0x400020116, 0x20, 0x800, 0x0}, // [1 2 4 8 17 34 69 139]
	/* idx: 140 */ {0x800020116, 0x40, 0x1000, 0x0}, // [1 2 4 8 17 35 70 140]
	/* idx: 141 */ {0x800020116, 0x40, 0x2000, 0x0}, // [1 2 4 8 17 35 70 141]
	/* idx: 142 */ {0x800020116, 0x80, 0x4000, 0x0}, // [1 2 4 8 17 35 71 142]
	/* idx: 143 */ {0x800020116, 0x80, 0x8000, 0x0}, // [1 2 4 8 17 35 71 143]
	/* idx: 144 */ {0x1000040216, 0x100, 0x10000, 0x0}, // [1 2 4 9 18 36 72 144]
	/* idx: 145 */ {0x1000040216, 0x100, 0x20000, 0x0}, // [1 2 4 9 18 36 72 145]
	/* idx: 146 */ {0x1000040216, 0x200, 0x40000, 0x0}, // [1 2 4 9 18 36 73 146]
	/* idx: 147 */ {0x1000040216, 0x200, 0x80000, 0x0}, // [1 2 4 9 18 36 73 147]
	/* idx: 148 */ {0x2000040216, 0x400, 0x100000, 0x0}, // [1 2 4 9 18 37 74 148]
	/* idx: 149 */ {0x2000040216, 0x400, 0x200000, 0x0}, // [1 2 4 9 18 37 74 149]
	/* idx: 150 */ {0x2000040216, 0x800, 0x400000, 0x0}, // [1 2 4 9 18 37 75 150]
	/* idx: 151 */ {0x2000040216, 0x800, 0x800000, 0x0}, // [1 2 4 9 18 37 75 151]
	/* idx: 152 */ {0x4000080216, 0x1000, 0x1000000, 0x0}, // [1 2 4 9 19 38 76 152]
	/* idx: 153 */ {0x4000080216, 0x1000, 0x2000000, 0x0}, // [1 2 4 9 19 38 76 153]
	/* idx: 154 */ {0x4000080216, 0x2000, 0x4000000, 0x0}, // [1 2 4 9 19 38 77 154]
	/* idx: 155 */ {0x4000080216, 0x2000, 0x8000000, 0x0}, // [1 2 4 9 19 38 77 155]
	/* idx: 156 */ {0x8000080216, 0x4000, 0x10000000, 0x0}, // [1 2 4 9 19 39 78 156]
	/* idx: 157 */ {0x8000080216, 0x4000, 0x20000000, 0x0}, // [1 2 4 9 19 39 78 157]
	/* idx: 158 */ {0x8000080216, 0x8000, 0x40000000, 0x0}, // [1 2 4 9 19 39 79 158]
	/* idx: 159 */ {0x8000080216, 0x8000, 0x80000000, 0x0}, // [1 2 4 9 19 39 79 159]
	/* idx: 160 */ {0x10000100426, 0x10000, 0x100000000, 0x0}, // [1 2 5 10 20 40 80 160]
	/* idx: 161 */ {0x10000100426, 0x10000, 0x200000000, 0x0}, // [1 2 5 10 20 40 80 161]
	/* idx: 162 */ {0x10000100426, 0x20000, 0x400000000, 0x0}, // [1 2 5 10 20 40 81 162]
	/* idx: 163 */ {0x10000100426, 0x20000, 0x800000000, 0x0}, // [1 2 5 10 20 40 81 163]
	/* idx: 164 */ {0x20000100426, 0x40000, 0x1000000000, 0x0}, // [1 2 5 10 20 41 82 164]
	/* idx: 165 */ {0x20000100426, 0x40000, 0x2000000000, 0x0}, // [1 2 5 10 20 41 82 165]
	/* idx: 166 */ {0x20000100426, 0x80000, 0x4000000000, 0x0}, // [1 2 5 10 20 41 83 166]
	/* idx: 167 */ {0x20000100426, 0x80000, 0x8000000000, 0x0}, // [1 2 5 10 20 41 83 167]
	/* idx: 168 */ {0x40000200426, 0x100000, 0x10000000000, 0x0}, // [1 2 5 10 21 42 84 168]
	/* idx: 169 */ {0x40000200426, 0x100000, 0x20000000000, 0x0}, // [1 2 5 10 21 42 84 169]
	/* idx: 170 */ {0x40000200426, 0x200000, 0x40000000000, 0x0}, // [1 2 5 10 21 42 85 170]
	/* idx: 171 */ {0x40000200426, 0x200000, 0x80000000000, 0x0}, // [1 2 5 10 21 42 85 171]
	/* idx: 172 */ {0x80000200426, 0x400000, 0x100000000000, 0x0}, // [1 2 5 10 21 43 86 172]
	/* idx: 173 */ {0x80000200426, 0x400000, 0x200000000000, 0x0}, // [1 2 5 10 21 43 86 173]
	/* idx: 174 */ {0x80000200426, 0x800000, 0x400000000000, 0x0}, // [1 2 5 10 21 43 87 174]
	/* idx: 175 */ {0x80000200426, 0x800000, 0x800000000000, 0x0}, // [1 2 5 10 21 43 87 175]
	/* idx: 176 */ {0x100000400826, 0x1000000, 0x1000000000000, 0x0}, // [1 2 5 11 22 44 88 176]
	/* idx: 177 */ {0x100000400826, 0x1000000, 0x2000000000000, 0x0}, // [1 2 5 11 22 44 88 177]
	/* idx: 178 */ {0x100000400826, 0x2000000, 0x4000000000000, 0x0}, // [1 2 5 11 22 44 89 178]
	/* idx: 179 */ {0x100000400826, 0x2000000, 0x8000000000000, 0x0}, // [1 2 5 11 22 44 89 179]
	/* idx: 180 */ {0x200000400826, 0x4000000, 0x10000000000000, 0x0}, // [1 2 5 11 22 45 90 180]
	/* idx: 181 */ {0x200000400826, 0x4000000, 0x20000000000000, 0x0}, // [1 2 5 11 22 45 90 181]
	/* idx: 182 */ {0x200000400826, 0x8000000, 0x40000000000000, 0x0}, // [1 2 5 11 22 45 91 182]
	/* idx: 183 */ {0x200000400826, 0x8000000, 0x80000000000000, 0x0}, // [1 2 5 11 22 45 91 183]
	/* idx: 184 */ {0x400000800826, 0x10000000, 0x100000000000000, 0x0}, // [1 2 5 11 23 46 92 184]
	/* idx: 185 */ {0x400000800826, 0x10000000, 0x200000000000000, 0x0}, // [1 2 5 11 23 46 92 185]
	/* idx: 186 */ {0x400000800826, 0x20000000, 0x400000000000000, 0x0}, // [1 2 5 11 23 46 93 186]
	/* idx: 187 */ {0x400000800826, 0x20000000, 0x800000000000000, 0x0}, // [1 2 5 11 23 46 93 187]
	/* idx: 188 */ {0x800000800826, 0x40000000, 0x1000000000000000, 0x0}, // [1 2 5 11 23 47 94 188]
	/* idx: 189 */ {0x800000800826, 0x40000000, 0x2000000000000000, 0x0}, // [1 2 5 11 23 47 94 189]
	/* idx: 190 */ {0x800000800826, 0x80000000, 0x4000000000000000, 0x0}, // [1 2 5 11 23 47 95 190]
	/* idx: 191 */ {0x800000800826, 0x80000000, 0x8000000000000000, 0x0}, // [1 2 5 11 23 47 95 191]
	/* idx: 192 */ {0x100000100104a, 0x100000000, 0x0, 0x1}, // [1 3 6 12 24 48 96 192]
	/* idx: 193 */ {0x100000100104a, 0x100000000, 0x0, 0x2}, // [1 3 6 12 24 48 96 193]
	/* idx: 194 */ {0x100000100104a, 0x200000000, 0x0, 0x4}, // [1 3 6 12 24 48 97 194]
	/* idx: 195 */ {0x100000100104a, 0x200000000, 0x0, 0x8}, // [1 3 6 12 24 48 97 195]
	/* idx: 196 */ {0x200000100104a, 0x400000000, 0x0, 0x10}, // [1 3 6 12 24 49 98 196]
	/* idx: 197 */ {0x200000100104a, 0x400000000, 0x0, 0x20}, // [1 3 6 12 24 49 98 197]
	/* idx: 198 */ {0x200000100104a, 0x800000000, 0x0, 0x40}, // [1 3 6 12 24 49 99 198]
	/* idx: 199 */ {0x200000100104a, 0x800000000, 0x0, 0x80}, // [1 3 6 12 24 49 99 199]
	/* idx: 200 */ {0x400000200104a, 0x1000000000, 0x0, 0x100}, // [1 3 6 12 25 50 100 200]
	/* idx: 201 */ {0x400000200104a, 0x1000000000, 0x0, 0x200}, // [1 3 6 12 25 50 100 201]
	/* idx: 202 */ {0x400000200104a, 0x2000000000, 0x0, 0x400}, // [1 3 6 12 25 50 101 202]
	/* idx: 203 */ {0x400000200104a, 0x2000000000, 0x0, 0x800}, // [1 3 6 12 25 50 101 203]
	/* idx: 204 */ {0x800000200104a, 0x4000000000, 0x0, 0x1000}, // [1 3 6 12 25 51 102 204]
	/* idx: 205 */ {0x800000200104a, 0x4000000000, 0x0, 0x2000}, // [1 3 6 12 25 51 102 205]
	/* idx: 206 */ {0x800000200104a, 0x8000000000, 0x0, 0x4000}, // [1 3 6 12 25 51 103 206]
	/* idx: 207 */ {0x800000200104a, 0x8000000000, 0x0, 0x8000}, // [1 3 6 12 25 51 103 207]
	/* idx: 208 */ {0x1000000400204a, 0x10000000000, 0x0, 0x10000}, // [1 3 6 13 26 52 104 208]
	/* idx: 209 */ {0x1000000400204a, 0x10000000000, 0x0, 0x20000}, // [1 3 6 13 26 52 104 209]
	/* idx: 210 */ {0x1000000400204a, 0x20000000000, 0x0, 0x40000}, // [1 3 6 13 26 52 105 210]
	/* idx: 211 */ {0x1000000400204a, 0x20000000000, 0x0, 0x80000}, // [1 3 6 13 26 52 105 211]
	/* idx: 212 */ {0x2000000400204a, 0x40000000000, 0x0, 0x100000}, // [1 3 6 13 26 53 106 212]
	/* idx: 213 */ {0x2000000400204a, 0x40000000000, 0x0, 0x200000}, // [1 3 6 13 26 53 106 213]
	/* idx: 214 */ {0x2000000400204a, 0x80000000000, 0x0, 0x400000}, // [1 3 6 13 26 53 107 214]
	/* idx: 215 */ {0x2000000400204a, 0x80000000000, 0x0, 0x800000}, // [1 3 6 13 26 53 107 215]
	/* idx: 216 */ {0x4000000800204a, 0x100000000000, 0x0, 0x1000000}, // [1 3 6 13 27 54 108 216]
	/* idx: 217 */ {0x4000000800204a, 0x100000000000, 0x0, 0x2000000}, // [1 3 6 13 27 54 108 217]
	/* idx: 218 */ {0x4000000800204a, 0x200000000000, 0x0, 0x4000000}, // [1 3 6 13 27 54 109 218]
	/* idx: 219 */ {0x4000000800204a, 0x200000000000, 0x0, 0x8000000}, // [1 3 6 13 27 54 109 219]
	/* idx: 220 */ {0x8000000800204a, 0x400000000000, 0x0, 0x10000000}, // [1 3 6 13 27 55 110 220]
	/* idx: 221 */ {0x8000000800204a, 0x400000000000, 0x0, 0x20000000}, // [1 3 6 13 27 55 110 221]
	/* idx: 222 */ {0x8000000800204a, 0x800000000000, 0x0, 0x40000000}, // [1 3 6 13 27 55 111 222]
	/* idx: 223 */ {0x8000000800204a, 0x800000000000, 0x0, 0x80000000}, // [1 3 6 13 27 55 111 223]
	/* idx: 224 */ {0x10000001000408a, 0x1000000000000, 0x0, 0x100000000}, // [1 3 7 14 28 56 112 224]
	/* idx: 225 */ {0x10000001000408a, 0x1000000000000, 0x0, 0x200000000}, // [1 3 7 14 28 56 112 225]
	/* idx: 226 */ {0x10000001000408a, 0x2000000000000, 0x0, 0x400000000}, // [1 3 7 14 28 56 113 226]
	/* idx: 227 */ {0x10000001000408a, 0x2000000000000, 0x0, 0x800000000}, // [1 3 7 14 28 56 113 227]
	/* idx: 228 */ {0x20000001000408a, 0x4000000000000, 0x0, 0x1000000000}, // [1 3 7 14 28 57 114 228]
	/* idx: 229 */ {0x20000001000408a, 0x4000000000000, 0x0, 0x2000000000}, // [1 3 7 14 28 57 114 229]
	/* idx: 230 */ {0x20000001000408a, 0x8000000000000, 0x0, 0x4000000000}, // [1 3 7 14 28 57 115 230]
	/* idx: 231 */ {0x20000001000408a, 0x8000000000000, 0x0, 0x8000000000}, // [1 3 7 14 28 57 115 231]
	/* idx: 232 */ {0x40000002000408a, 0x10000000000000, 0x0, 0x10000000000}, // [1 3 7 14 29 58 116 232]
	/* idx: 233 */ {0x40000002000408a, 0x10000000000000, 0x0, 0x20000000000}, // [1 3 7 14 29 58 116 233]
	/* idx: 234 */ {0x40000002000408a, 0x20000000000000, 0x0, 0x40000000000}, // [1 3 7 14 29 58 117 234]
	/* idx: 235 */ {0x40000002000408a, 0x20000000000000, 0x0, 0x80000000000}, // [1 3 7 14 29 58 117 235]
	/* idx: 236 */ {0x80000002000408a, 0x40000000000000, 0x0, 0x100000000000}, // [1 3 7 14 29 59 118 236]
	/* idx: 237 */ {0x80000002000408a, 0x40000000000000, 0x0, 0x200000000000}, // [1 3 7 14 29 59 118 237]
	/* idx: 238 */ {0x80000002000408a, 0x80000000000000, 0x0, 0x400000000000}, // [1 3 7 14 29 59 119 238]
	/* idx: 239 */ {0x80000002000408a, 0x80000000000000, 0x0, 0x800000000000}, // [1 3 7 14 29 59 119 239]
	/* idx: 240 */ {0x100000004000808a, 0x100000000000000, 0x0, 0x1000000000000}, // [1 3 7 15 30 60 120 240]
	/* idx: 241 */ {0x100000004000808a, 0x100000000000000, 0x0, 0x2000000000000}, // [1 3 7 15 30 60 120 241]
	/* idx: 242 */ {0x100000004000808a, 0x200000000000000, 0x0, 0x4000000000000}, // [1 3 7 15 30 60 121 242]
	/* idx: 243 */ {0x100000004000808a, 0x200000000000000, 0x0, 0x8000000000000}, // [1 3 7 15 30 60 121 243]
	/* idx: 244 */ {0x200000004000808a, 0x400000000000000, 0x0, 0x10000000000000}, // [1 3 7 15 30 61 122 244]
	/* idx: 245 */ {0x200000004000808a, 0x400000000000000, 0x0, 0x20000000000000}, // [1 3 7 15 30 61 122 245]
	/* idx: 246 */ {0x200000004000808a, 0x800000000000000, 0x0, 0x40000000000000}, // [1 3 7 15 30 61 123 246]
	/* idx: 247 */ {0x200000004000808a, 0x800000000000000, 0x0, 0x80000000000000}, // [1 3 7 15 30 61 123 247]
	/* idx: 248 */ {0x400000008000808a, 0x1000000000000000, 0x0, 0x100000000000000}, // [1 3 7 15 31 62 124 248]
	/* idx: 249 */ {0x400000008000808a, 0x1000000000000000, 0x0, 0x200000000000000}, // [1 3 7 15 31 62 124 249]
	/* idx: 250 */ {0x400000008000808a, 0x2000000000000000, 0x0, 0x400000000000000}, // [1 3 7 15 31 62 125 250]
	/* idx: 251 */ {0x400000008000808a, 0x2000000000000000, 0x0, 0x800000000000000}, // [1 3 7 15 31 62 125 251]
	/* idx: 252 */ {0x800000008000808a, 0x4000000000000000, 0x0, 0x1000000000000000}, // [1 3 7 15 31 63 126 252]
	/* idx: 253 */ {0x800000008000808a, 0x4000000000000000, 0x0, 0x2000000000000000}, // [1 3 7 15 31 63 126 253]
	/* idx: 254 */ {0x800000008000808a, 0x8000000000000000, 0x0, 0x4000000000000000}, // [1 3 7 15 31 63 127 254]
	/* idx: 255 */ {0x800000008000808a, 0x8000000000000000, 0x0, 0x8000000000000000}, // [1 3 7 15 31 63 127 255]
	//
	// ###################################################################
	//  the mapping funktion: f(idx) -> lpm-backtracking-bitset
	//  is restricted to [4]uint64 == 256 bit as result
	// ###################################################################
	//
	/* idx: 256 */ {0x100010116, 0x1, 0x1, 0x0}, // [1 2 4 8 16 32 64 128]
	/* idx: 257 */ {0x100010116, 0x1, 0x1, 0x0}, // [1 2 4 8 16 32 64 128]
	/* idx: 258 */ {0x100010116, 0x1, 0x2, 0x0}, // [1 2 4 8 16 32 64 129]
	/* idx: 259 */ {0x100010116, 0x1, 0x2, 0x0}, // [1 2 4 8 16 32 64 129]
	/* idx: 260 */ {0x100010116, 0x2, 0x4, 0x0}, // [1 2 4 8 16 32 65 130]
	/* idx: 261 */ {0x100010116, 0x2, 0x4, 0x0}, // [1 2 4 8 16 32 65 130]
	/* idx: 262 */ {0x100010116, 0x2, 0x8, 0x0}, // [1 2 4 8 16 32 65 131]
	/* idx: 263 */ {0x100010116, 0x2, 0x8, 0x0}, // [1 2 4 8 16 32 65 131]
	/* idx: 264 */ {0x200010116, 0x4, 0x10, 0x0}, // [1 2 4 8 16 33 66 132]
	/* idx: 265 */ {0x200010116, 0x4, 0x10, 0x0}, // [1 2 4 8 16 33 66 132]
	/* idx: 266 */ {0x200010116, 0x4, 0x20, 0x0}, // [1 2 4 8 16 33 66 133]
	/* idx: 267 */ {0x200010116, 0x4, 0x20, 0x0}, // [1 2 4 8 16 33 66 133]
	/* idx: 268 */ {0x200010116, 0x8, 0x40, 0x0}, // [1 2 4 8 16 33 67 134]
	/* idx: 269 */ {0x200010116, 0x8, 0x40, 0x0}, // [1 2 4 8 16 33 67 134]
	/* idx: 270 */ {0x200010116, 0x8, 0x80, 0x0}, // [1 2 4 8 16 33 67 135]
	/* idx: 271 */ {0x200010116, 0x8, 0x80, 0x0}, // [1 2 4 8 16 33 67 135]
	/* idx: 272 */ {0x400020116, 0x10, 0x100, 0x0}, // [1 2 4 8 17 34 68 136]
	/* idx: 273 */ {0x400020116, 0x10, 0x100, 0x0}, // [1 2 4 8 17 34 68 136]
	/* idx: 274 */ {0x400020116, 0x10, 0x200, 0x0}, // [1 2 4 8 17 34 68 137]
	/* idx: 275 */ {0x400020116, 0x10, 0x200, 0x0}, // [1 2 4 8 17 34 68 137]
	/* idx: 276 */ {0x400020116, 0x20, 0x400, 0x0}, // [1 2 4 8 17 34 69 138]
	/* idx: 277 */ {0x400020116, 0x20, 0x400, 0x0}, // [1 2 4 8 17 34 69 138]
	/* idx: 278 */ {0x400020116, 0x20, 0x800, 0x0}, // [1 2 4 8 17 34 69 139]
	/* idx: 279 */ {0x400020116, 0x20, 0x800, 0x0}, // [1 2 4 8 17 34 69 139]
	/* idx: 280 */ {0x800020116, 0x40, 0x1000, 0x0}, // [1 2 4 8 17 35 70 140]
	/* idx: 281 */ {0x800020116, 0x40, 0x1000, 0x0}, // [1 2 4 8 17 35 70 140]
	/* idx: 282 */ {0x800020116, 0x40, 0x2000, 0x0}, // [1 2 4 8 17 35 70 141]
	/* idx: 283 */ {0x800020116, 0x40, 0x2000, 0x0}, // [1 2 4 8 17 35 70 141]
	/* idx: 284 */ {0x800020116, 0x80, 0x4000, 0x0}, // [1 2 4 8 17 35 71 142]
	/* idx: 285 */ {0x800020116, 0x80, 0x4000, 0x0}, // [1 2 4 8 17 35 71 142]
	/* idx: 286 */ {0x800020116, 0x80, 0x8000, 0x0}, // [1 2 4 8 17 35 71 143]
	/* idx: 287 */ {0x800020116, 0x80, 0x8000, 0x0}, // [1 2 4 8 17 35 71 143]
	/* idx: 288 */ {0x1000040216, 0x100, 0x10000, 0x0}, // [1 2 4 9 18 36 72 144]
	/* idx: 289 */ {0x1000040216, 0x100, 0x10000, 0x0}, // [1 2 4 9 18 36 72 144]
	/* idx: 290 */ {0x1000040216, 0x100, 0x20000, 0x0}, // [1 2 4 9 18 36 72 145]
	/* idx: 291 */ {0x1000040216, 0x100, 0x20000, 0x0}, // [1 2 4 9 18 36 72 145]
	/* idx: 292 */ {0x1000040216, 0x200, 0x40000, 0x0}, // [1 2 4 9 18 36 73 146]
	/* idx: 293 */ {0x1000040216, 0x200, 0x40000, 0x0}, // [1 2 4 9 18 36 73 146]
	/* idx: 294 */ {0x1000040216, 0x200, 0x80000, 0x0}, // [1 2 4 9 18 36 73 147]
	/* idx: 295 */ {0x1000040216, 0x200, 0x80000, 0x0}, // [1 2 4 9 18 36 73 147]
	/* idx: 296 */ {0x2000040216, 0x400, 0x100000, 0x0}, // [1 2 4 9 18 37 74 148]
	/* idx: 297 */ {0x2000040216, 0x400, 0x100000, 0x0}, // [1 2 4 9 18 37 74 148]
	/* idx: 298 */ {0x2000040216, 0x400, 0x200000, 0x0}, // [1 2 4 9 18 37 74 149]
	/* idx: 299 */ {0x2000040216, 0x400, 0x200000, 0x0}, // [1 2 4 9 18 37 74 149]
	/* idx: 300 */ {0x2000040216, 0x800, 0x400000, 0x0}, // [1 2 4 9 18 37 75 150]
	/* idx: 301 */ {0x2000040216, 0x800, 0x400000, 0x0}, // [1 2 4 9 18 37 75 150]
	/* idx: 302 */ {0x2000040216, 0x800, 0x800000, 0x0}, // [1 2 4 9 18 37 75 151]
	/* idx: 303 */ {0x2000040216, 0x800, 0x800000, 0x0}, // [1 2 4 9 18 37 75 151]
	/* idx: 304 */ {0x4000080216, 0x1000, 0x1000000, 0x0}, // [1 2 4 9 19 38 76 152]
	/* idx: 305 */ {0x4000080216, 0x1000, 0x1000000, 0x0}, // [1 2 4 9 19 38 76 152]
	/* idx: 306 */ {0x4000080216, 0x1000, 0x2000000, 0x0}, // [1 2 4 9 19 38 76 153]
	/* idx: 307 */ {0x4000080216, 0x1000, 0x2000000, 0x0}, // [1 2 4 9 19 38 76 153]
	/* idx: 308 */ {0x4000080216, 0x2000, 0x4000000, 0x0}, // [1 2 4 9 19 38 77 154]
	/* idx: 309 */ {0x4000080216, 0x2000, 0x4000000, 0x0}, // [1 2 4 9 19 38 77 154]
	/* idx: 310 */ {0x4000080216, 0x2000, 0x8000000, 0x0}, // [1 2 4 9 19 38 77 155]
	/* idx: 311 */ {0x4000080216, 0x2000, 0x8000000, 0x0}, // [1 2 4 9 19 38 77 155]
	/* idx: 312 */ {0x8000080216, 0x4000, 0x10000000, 0x0}, // [1 2 4 9 19 39 78 156]
	/* idx: 313 */ {0x8000080216, 0x4000, 0x10000000, 0x0}, // [1 2 4 9 19 39 78 156]
	/* idx: 314 */ {0x8000080216, 0x4000, 0x20000000, 0x0}, // [1 2 4 9 19 39 78 157]
	/* idx: 315 */ {0x8000080216, 0x4000, 0x20000000, 0x0}, // [1 2 4 9 19 39 78 157]
	/* idx: 316 */ {0x8000080216, 0x8000, 0x40000000, 0x0}, // [1 2 4 9 19 39 79 158]
	/* idx: 317 */ {0x8000080216, 0x8000, 0x40000000, 0x0}, // [1 2 4 9 19 39 79 158]
	/* idx: 318 */ {0x8000080216, 0x8000, 0x80000000, 0x0}, // [1 2 4 9 19 39 79 159]
	/* idx: 319 */ {0x8000080216, 0x8000, 0x80000000, 0x0}, // [1 2 4 9 19 39 79 159]
	/* idx: 320 */ {0x10000100426, 0x10000, 0x100000000, 0x0}, // [1 2 5 10 20 40 80 160]
	/* idx: 321 */ {0x10000100426, 0x10000, 0x100000000, 0x0}, // [1 2 5 10 20 40 80 160]
	/* idx: 322 */ {0x10000100426, 0x10000, 0x200000000, 0x0}, // [1 2 5 10 20 40 80 161]
	/* idx: 323 */ {0x10000100426, 0x10000, 0x200000000, 0x0}, // [1 2 5 10 20 40 80 161]
	/* idx: 324 */ {0x10000100426, 0x20000, 0x400000000, 0x0}, // [1 2 5 10 20 40 81 162]
	/* idx: 325 */ {0x10000100426, 0x20000, 0x400000000, 0x0}, // [1 2 5 10 20 40 81 162]
	/* idx: 326 */ {0x10000100426, 0x20000, 0x800000000, 0x0}, // [1 2 5 10 20 40 81 163]
	/* idx: 327 */ {0x10000100426, 0x20000, 0x800000000, 0x0}, // [1 2 5 10 20 40 81 163]
	/* idx: 328 */ {0x20000100426, 0x40000, 0x1000000000, 0x0}, // [1 2 5 10 20 41 82 164]
	/* idx: 329 */ {0x20000100426, 0x40000, 0x1000000000, 0x0}, // [1 2 5 10 20 41 82 164]
	/* idx: 330 */ {0x20000100426, 0x40000, 0x2000000000, 0x0}, // [1 2 5 10 20 41 82 165]
	/* idx: 331 */ {0x20000100426, 0x40000, 0x2000000000, 0x0}, // [1 2 5 10 20 41 82 165]
	/* idx: 332 */ {0x20000100426, 0x80000, 0x4000000000, 0x0}, // [1 2 5 10 20 41 83 166]
	/* idx: 333 */ {0x20000100426, 0x80000, 0x4000000000, 0x0}, // [1 2 5 10 20 41 83 166]
	/* idx: 334 */ {0x20000100426, 0x80000, 0x8000000000, 0x0}, // [1 2 5 10 20 41 83 167]
	/* idx: 335 */ {0x20000100426, 0x80000, 0x8000000000, 0x0}, // [1 2 5 10 20 41 83 167]
	/* idx: 336 */ {0x40000200426, 0x100000, 0x10000000000, 0x0}, // [1 2 5 10 21 42 84 168]
	/* idx: 337 */ {0x40000200426, 0x100000, 0x10000000000, 0x0}, // [1 2 5 10 21 42 84 168]
	/* idx: 338 */ {0x40000200426, 0x100000, 0x20000000000, 0x0}, // [1 2 5 10 21 42 84 169]
	/* idx: 339 */ {0x40000200426, 0x100000, 0x20000000000, 0x0}, // [1 2 5 10 21 42 84 169]
	/* idx: 340 */ {0x40000200426, 0x200000, 0x40000000000, 0x0}, // [1 2 5 10 21 42 85 170]
	/* idx: 341 */ {0x40000200426, 0x200000, 0x40000000000, 0x0}, // [1 2 5 10 21 42 85 170]
	/* idx: 342 */ {0x40000200426, 0x200000, 0x80000000000, 0x0}, // [1 2 5 10 21 42 85 171]
	/* idx: 343 */ {0x40000200426, 0x200000, 0x80000000000, 0x0}, // [1 2 5 10 21 42 85 171]
	/* idx: 344 */ {0x80000200426, 0x400000, 0x100000000000, 0x0}, // [1 2 5 10 21 43 86 172]
	/* idx: 345 */ {0x80000200426, 0x400000, 0x100000000000, 0x0}, // [1 2 5 10 21 43 86 172]
	/* idx: 346 */ {0x80000200426, 0x400000, 0x200000000000, 0x0}, // [1 2 5 10 21 43 86 173]
	/* idx: 347 */ {0x80000200426, 0x400000, 0x200000000000, 0x0}, // [1 2 5 10 21 43 86 173]
	/* idx: 348 */ {0x80000200426, 0x800000, 0x400000000000, 0x0}, // [1 2 5 10 21 43 87 174]
	/* idx: 349 */ {0x80000200426, 0x800000, 0x400000000000, 0x0}, // [1 2 5 10 21 43 87 174]
	/* idx: 350 */ {0x80000200426, 0x800000, 0x800000000000, 0x0}, // [1 2 5 10 21 43 87 175]
	/* idx: 351 */ {0x80000200426, 0x800000, 0x800000000000, 0x0}, // [1 2 5 10 21 43 87 175]
	/* idx: 352 */ {0x100000400826, 0x1000000, 0x1000000000000, 0x0}, // [1 2 5 11 22 44 88 176]
	/* idx: 353 */ {0x100000400826, 0x1000000, 0x1000000000000, 0x0}, // [1 2 5 11 22 44 88 176]
	/* idx: 354 */ {0x100000400826, 0x1000000, 0x2000000000000, 0x0}, // [1 2 5 11 22 44 88 177]
	/* idx: 355 */ {0x100000400826, 0x1000000, 0x2000000000000, 0x0}, // [1 2 5 11 22 44 88 177]
	/* idx: 356 */ {0x100000400826, 0x2000000, 0x4000000000000, 0x0}, // [1 2 5 11 22 44 89 178]
	/* idx: 357 */ {0x100000400826, 0x2000000, 0x4000000000000, 0x0}, // [1 2 5 11 22 44 89 178]
	/* idx: 358 */ {0x100000400826, 0x2000000, 0x8000000000000, 0x0}, // [1 2 5 11 22 44 89 179]
	/* idx: 359 */ {0x100000400826, 0x2000000, 0x8000000000000, 0x0}, // [1 2 5 11 22 44 89 179]
	/* idx: 360 */ {0x200000400826, 0x4000000, 0x10000000000000, 0x0}, // [1 2 5 11 22 45 90 180]
	/* idx: 361 */ {0x200000400826, 0x4000000, 0x10000000000000, 0x0}, // [1 2 5 11 22 45 90 180]
	/* idx: 362 */ {0x200000400826, 0x4000000, 0x20000000000000, 0x0}, // [1 2 5 11 22 45 90 181]
	/* idx: 363 */ {0x200000400826, 0x4000000, 0x20000000000000, 0x0}, // [1 2 5 11 22 45 90 181]
	/* idx: 364 */ {0x200000400826, 0x8000000, 0x40000000000000, 0x0}, // [1 2 5 11 22 45 91 182]
	/* idx: 365 */ {0x200000400826, 0x8000000, 0x40000000000000, 0x0}, // [1 2 5 11 22 45 91 182]
	/* idx: 366 */ {0x200000400826, 0x8000000, 0x80000000000000, 0x0}, // [1 2 5 11 22 45 91 183]
	/* idx: 367 */ {0x200000400826, 0x8000000, 0x80000000000000, 0x0}, // [1 2 5 11 22 45 91 183]
	/* idx: 368 */ {0x400000800826, 0x10000000, 0x100000000000000, 0x0}, // [1 2 5 11 23 46 92 184]
	/* idx: 369 */ {0x400000800826, 0x10000000, 0x100000000000000, 0x0}, // [1 2 5 11 23 46 92 184]
	/* idx: 370 */ {0x400000800826, 0x10000000, 0x200000000000000, 0x0}, // [1 2 5 11 23 46 92 185]
	/* idx: 371 */ {0x400000800826, 0x10000000, 0x200000000000000, 0x0}, // [1 2 5 11 23 46 92 185]
	/* idx: 372 */ {0x400000800826, 0x20000000, 0x400000000000000, 0x0}, // [1 2 5 11 23 46 93 186]
	/* idx: 373 */ {0x400000800826, 0x20000000, 0x400000000000000, 0x0}, // [1 2 5 11 23 46 93 186]
	/* idx: 374 */ {0x400000800826, 0x20000000, 0x800000000000000, 0x0}, // [1 2 5 11 23 46 93 187]
	/* idx: 375 */ {0x400000800826, 0x20000000, 0x800000000000000, 0x0}, // [1 2 5 11 23 46 93 187]
	/* idx: 376 */ {0x800000800826, 0x40000000, 0x1000000000000000, 0x0}, // [1 2 5 11 23 47 94 188]
	/* idx: 377 */ {0x800000800826, 0x40000000, 0x1000000000000000, 0x0}, // [1 2 5 11 23 47 94 188]
	/* idx: 378 */ {0x800000800826, 0x40000000, 0x2000000000000000, 0x0}, // [1 2 5 11 23 47 94 189]
	/* idx: 379 */ {0x800000800826, 0x40000000, 0x2000000000000000, 0x0}, // [1 2 5 11 23 47 94 189]
	/* idx: 380 */ {0x800000800826, 0x80000000, 0x4000000000000000, 0x0}, // [1 2 5 11 23 47 95 190]
	/* idx: 381 */ {0x800000800826, 0x80000000, 0x4000000000000000, 0x0}, // [1 2 5 11 23 47 95 190]
	/* idx: 382 */ {0x800000800826, 0x80000000, 0x8000000000000000, 0x0}, // [1 2 5 11 23 47 95 191]
	/* idx: 383 */ {0x800000800826, 0x80000000, 0x8000000000000000, 0x0}, // [1 2 5 11 23 47 95 191]
	/* idx: 384 */ {0x100000100104a, 0x100000000, 0x0, 0x1}, // [1 3 6 12 24 48 96 192]
	/* idx: 385 */ {0x100000100104a, 0x100000000, 0x0, 0x1}, // [1 3 6 12 24 48 96 192]
	/* idx: 386 */ {0x100000100104a, 0x100000000, 0x0, 0x2}, // [1 3 6 12 24 48 96 193]
	/* idx: 387 */ {0x100000100104a, 0x100000000, 0x0, 0x2}, // [1 3 6 12 24 48 96 193]
	/* idx: 388 */ {0x100000100104a, 0x200000000, 0x0, 0x4}, // [1 3 6 12 24 48 97 194]
	/* idx: 389 */ {0x100000100104a, 0x200000000, 0x0, 0x4}, // [1 3 6 12 24 48 97 194]
	/* idx: 390 */ {0x100000100104a, 0x200000000, 0x0, 0x8}, // [1 3 6 12 24 48 97 195]
	/* idx: 391 */ {0x100000100104a, 0x200000000, 0x0, 0x8}, // [1 3 6 12 24 48 97 195]
	/* idx: 392 */ {0x200000100104a, 0x400000000, 0x0, 0x10}, // [1 3 6 12 24 49 98 196]
	/* idx: 393 */ {0x200000100104a, 0x400000000, 0x0, 0x10}, // [1 3 6 12 24 49 98 196]
	/* idx: 394 */ {0x200000100104a, 0x400000000, 0x0, 0x20}, // [1 3 6 12 24 49 98 197]
	/* idx: 395 */ {0x200000100104a, 0x400000000, 0x0, 0x20}, // [1 3 6 12 24 49 98 197]
	/* idx: 396 */ {0x200000100104a, 0x800000000, 0x0, 0x40}, // [1 3 6 12 24 49 99 198]
	/* idx: 397 */ {0x200000100104a, 0x800000000, 0x0, 0x40}, // [1 3 6 12 24 49 99 198]
	/* idx: 398 */ {0x200000100104a, 0x800000000, 0x0, 0x80}, // [1 3 6 12 24 49 99 199]
	/* idx: 399 */ {0x200000100104a, 0x800000000, 0x0, 0x80}, // [1 3 6 12 24 49 99 199]
	/* idx: 400 */ {0x400000200104a, 0x1000000000, 0x0, 0x100}, // [1 3 6 12 25 50 100 200]
	/* idx: 401 */ {0x400000200104a, 0x1000000000, 0x0, 0x100}, // [1 3 6 12 25 50 100 200]
	/* idx: 402 */ {0x400000200104a, 0x1000000000, 0x0, 0x200}, // [1 3 6 12 25 50 100 201]
	/* idx: 403 */ {0x400000200104a, 0x1000000000, 0x0, 0x200}, // [1 3 6 12 25 50 100 201]
	/* idx: 404 */ {0x400000200104a, 0x2000000000, 0x0, 0x400}, // [1 3 6 12 25 50 101 202]
	/* idx: 405 */ {0x400000200104a, 0x2000000000, 0x0, 0x400}, // [1 3 6 12 25 50 101 202]
	/* idx: 406 */ {0x400000200104a, 0x2000000000, 0x0, 0x800}, // [1 3 6 12 25 50 101 203]
	/* idx: 407 */ {0x400000200104a, 0x2000000000, 0x0, 0x800}, // [1 3 6 12 25 50 101 203]
	/* idx: 408 */ {0x800000200104a, 0x4000000000, 0x0, 0x1000}, // [1 3 6 12 25 51 102 204]
	/* idx: 409 */ {0x800000200104a, 0x4000000000, 0x0, 0x1000}, // [1 3 6 12 25 51 102 204]
	/* idx: 410 */ {0x800000200104a, 0x4000000000, 0x0, 0x2000}, // [1 3 6 12 25 51 102 205]
	/* idx: 411 */ {0x800000200104a, 0x4000000000, 0x0, 0x2000}, // [1 3 6 12 25 51 102 205]
	/* idx: 412 */ {0x800000200104a, 0x8000000000, 0x0, 0x4000}, // [1 3 6 12 25 51 103 206]
	/* idx: 413 */ {0x800000200104a, 0x8000000000, 0x0, 0x4000}, // [1 3 6 12 25 51 103 206]
	/* idx: 414 */ {0x800000200104a, 0x8000000000, 0x0, 0x8000}, // [1 3 6 12 25 51 103 207]
	/* idx: 415 */ {0x800000200104a, 0x8000000000, 0x0, 0x8000}, // [1 3 6 12 25 51 103 207]
	/* idx: 416 */ {0x1000000400204a, 0x10000000000, 0x0, 0x10000}, // [1 3 6 13 26 52 104 208]
	/* idx: 417 */ {0x1000000400204a, 0x10000000000, 0x0, 0x10000}, // [1 3 6 13 26 52 104 208]
	/* idx: 418 */ {0x1000000400204a, 0x10000000000, 0x0, 0x20000}, // [1 3 6 13 26 52 104 209]
	/* idx: 419 */ {0x1000000400204a, 0x10000000000, 0x0, 0x20000}, // [1 3 6 13 26 52 104 209]
	/* idx: 420 */ {0x1000000400204a, 0x20000000000, 0x0, 0x40000}, // [1 3 6 13 26 52 105 210]
	/* idx: 421 */ {0x1000000400204a, 0x20000000000, 0x0, 0x40000}, // [1 3 6 13 26 52 105 210]
	/* idx: 422 */ {0x1000000400204a, 0x20000000000, 0x0, 0x80000}, // [1 3 6 13 26 52 105 211]
	/* idx: 423 */ {0x1000000400204a, 0x20000000000, 0x0, 0x80000}, // [1 3 6 13 26 52 105 211]
	/* idx: 424 */ {0x2000000400204a, 0x40000000000, 0x0, 0x100000}, // [1 3 6 13 26 53 106 212]
	/* idx: 425 */ {0x2000000400204a, 0x40000000000, 0x0, 0x100000}, // [1 3 6 13 26 53 106 212]
	/* idx: 426 */ {0x2000000400204a, 0x40000000000, 0x0, 0x200000}, // [1 3 6 13 26 53 106 213]
	/* idx: 427 */ {0x2000000400204a, 0x40000000000, 0x0, 0x200000}, // [1 3 6 13 26 53 106 213]
	/* idx: 428 */ {0x2000000400204a, 0x80000000000, 0x0, 0x400000}, // [1 3 6 13 26 53 107 214]
	/* idx: 429 */ {0x2000000400204a, 0x80000000000, 0x0, 0x400000}, // [1 3 6 13 26 53 107 214]
	/* idx: 430 */ {0x2000000400204a, 0x80000000000, 0x0, 0x800000}, // [1 3 6 13 26 53 107 215]
	/* idx: 431 */ {0x2000000400204a, 0x80000000000, 0x0, 0x800000}, // [1 3 6 13 26 53 107 215]
	/* idx: 432 */ {0x4000000800204a, 0x100000000000, 0x0, 0x1000000}, // [1 3 6 13 27 54 108 216]
	/* idx: 433 */ {0x4000000800204a, 0x100000000000, 0x0, 0x1000000}, // [1 3 6 13 27 54 108 216]
	/* idx: 434 */ {0x4000000800204a, 0x100000000000, 0x0, 0x2000000}, // [1 3 6 13 27 54 108 217]
	/* idx: 435 */ {0x4000000800204a, 0x100000000000, 0x0, 0x2000000}, // [1 3 6 13 27 54 108 217]
	/* idx: 436 */ {0x4000000800204a, 0x200000000000, 0x0, 0x4000000}, // [1 3 6 13 27 54 109 218]
	/* idx: 437 */ {0x4000000800204a, 0x200000000000, 0x0, 0x4000000}, // [1 3 6 13 27 54 109 218]
	/* idx: 438 */ {0x4000000800204a, 0x200000000000, 0x0, 0x8000000}, // [1 3 6 13 27 54 109 219]
	/* idx: 439 */ {0x4000000800204a, 0x200000000000, 0x0, 0x8000000}, // [1 3 6 13 27 54 109 219]
	/* idx: 440 */ {0x8000000800204a, 0x400000000000, 0x0, 0x10000000}, // [1 3 6 13 27 55 110 220]
	/* idx: 441 */ {0x8000000800204a, 0x400000000000, 0x0, 0x10000000}, // [1 3 6 13 27 55 110 220]
	/* idx: 442 */ {0x8000000800204a, 0x400000000000, 0x0, 0x20000000}, // [1 3 6 13 27 55 110 221]
	/* idx: 443 */ {0x8000000800204a, 0x400000000000, 0x0, 0x20000000}, // [1 3 6 13 27 55 110 221]
	/* idx: 444 */ {0x8000000800204a, 0x800000000000, 0x0, 0x40000000}, // [1 3 6 13 27 55 111 222]
	/* idx: 445 */ {0x8000000800204a, 0x800000000000, 0x0, 0x40000000}, // [1 3 6 13 27 55 111 222]
	/* idx: 446 */ {0x8000000800204a, 0x800000000000, 0x0, 0x80000000}, // [1 3 6 13 27 55 111 223]
	/* idx: 447 */ {0x8000000800204a, 0x800000000000, 0x0, 0x80000000}, // [1 3 6 13 27 55 111 223]
	/* idx: 448 */ {0x10000001000408a, 0x1000000000000, 0x0, 0x100000000}, // [1 3 7 14 28 56 112 224]
	/* idx: 449 */ {0x10000001000408a, 0x1000000000000, 0x0, 0x100000000}, // [1 3 7 14 28 56 112 224]
	/* idx: 450 */ {0x10000001000408a, 0x1000000000000, 0x0, 0x200000000}, // [1 3 7 14 28 56 112 225]
	/* idx: 451 */ {0x10000001000408a, 0x1000000000000, 0x0, 0x200000000}, // [1 3 7 14 28 56 112 225]
	/* idx: 452 */ {0x10000001000408a, 0x2000000000000, 0x0, 0x400000000}, // [1 3 7 14 28 56 113 226]
	/* idx: 453 */ {0x10000001000408a, 0x2000000000000, 0x0, 0x400000000}, // [1 3 7 14 28 56 113 226]
	/* idx: 454 */ {0x10000001000408a, 0x2000000000000, 0x0, 0x800000000}, // [1 3 7 14 28 56 113 227]
	/* idx: 455 */ {0x10000001000408a, 0x2000000000000, 0x0, 0x800000000}, // [1 3 7 14 28 56 113 227]
	/* idx: 456 */ {0x20000001000408a, 0x4000000000000, 0x0, 0x1000000000}, // [1 3 7 14 28 57 114 228]
	/* idx: 457 */ {0x20000001000408a, 0x4000000000000, 0x0, 0x1000000000}, // [1 3 7 14 28 57 114 228]
	/* idx: 458 */ {0x20000001000408a, 0x4000000000000, 0x0, 0x2000000000}, // [1 3 7 14 28 57 114 229]
	/* idx: 459 */ {0x20000001000408a, 0x4000000000000, 0x0, 0x2000000000}, // [1 3 7 14 28 57 114 229]
	/* idx: 460 */ {0x20000001000408a, 0x8000000000000, 0x0, 0x4000000000}, // [1 3 7 14 28 57 115 230]
	/* idx: 461 */ {0x20000001000408a, 0x8000000000000, 0x0, 0x4000000000}, // [1 3 7 14 28 57 115 230]
	/* idx: 462 */ {0x20000001000408a, 0x8000000000000, 0x0, 0x8000000000}, // [1 3 7 14 28 57 115 231]
	/* idx: 463 */ {0x20000001000408a, 0x8000000000000, 0x0, 0x8000000000}, // [1 3 7 14 28 57 115 231]
	/* idx: 464 */ {0x40000002000408a, 0x10000000000000, 0x0, 0x10000000000}, // [1 3 7 14 29 58 116 232]
	/* idx: 465 */ {0x40000002000408a, 0x10000000000000, 0x0, 0x10000000000}, // [1 3 7 14 29 58 116 232]
	/* idx: 466 */ {0x40000002000408a, 0x10000000000000, 0x0, 0x20000000000}, // [1 3 7 14 29 58 116 233]
	/* idx: 467 */ {0x40000002000408a, 0x10000000000000, 0x0, 0x20000000000}, // [1 3 7 14 29 58 116 233]
	/* idx: 468 */ {0x40000002000408a, 0x20000000000000, 0x0, 0x40000000000}, // [1 3 7 14 29 58 117 234]
	/* idx: 469 */ {0x40000002000408a, 0x20000000000000, 0x0, 0x40000000000}, // [1 3 7 14 29 58 117 234]
	/* idx: 470 */ {0x40000002000408a, 0x20000000000000, 0x0, 0x80000000000}, // [1 3 7 14 29 58 117 235]
	/* idx: 471 */ {0x40000002000408a, 0x20000000000000, 0x0, 0x80000000000}, // [1 3 7 14 29 58 117 235]
	/* idx: 472 */ {0x80000002000408a, 0x40000000000000, 0x0, 0x100000000000}, // [1 3 7 14 29 59 118 236]
	/* idx: 473 */ {0x80000002000408a, 0x40000000000000, 0x0, 0x100000000000}, // [1 3 7 14 29 59 118 236]
	/* idx: 474 */ {0x80000002000408a, 0x40000000000000, 0x0, 0x200000000000}, // [1 3 7 14 29 59 118 237]
	/* idx: 475 */ {0x80000002000408a, 0x40000000000000, 0x0, 0x200000000000}, // [1 3 7 14 29 59 118 237]
	/* idx: 476 */ {0x80000002000408a, 0x80000000000000, 0x0, 0x400000000000}, // [1 3 7 14 29 59 119 238]
	/* idx: 477 */ {0x80000002000408a, 0x80000000000000, 0x0, 0x400000000000}, // [1 3 7 14 29 59 119 238]
	/* idx: 478 */ {0x80000002000408a, 0x80000000000000, 0x0, 0x800000000000}, // [1 3 7 14 29 59 119 239]
	/* idx: 479 */ {0x80000002000408a, 0x80000000000000, 0x0, 0x800000000000}, // [1 3 7 14 29 59 119 239]
	/* idx: 480 */ {0x100000004000808a, 0x100000000000000, 0x0, 0x1000000000000}, // [1 3 7 15 30 60 120 240]
	/* idx: 481 */ {0x100000004000808a, 0x100000000000000, 0x0, 0x1000000000000}, // [1 3 7 15 30 60 120 240]
	/* idx: 482 */ {0x100000004000808a, 0x100000000000000, 0x0, 0x2000000000000}, // [1 3 7 15 30 60 120 241]
	/* idx: 483 */ {0x100000004000808a, 0x100000000000000, 0x0, 0x2000000000000}, // [1 3 7 15 30 60 120 241]
	/* idx: 484 */ {0x100000004000808a, 0x200000000000000, 0x0, 0x4000000000000}, // [1 3 7 15 30 60 121 242]
	/* idx: 485 */ {0x100000004000808a, 0x200000000000000, 0x0, 0x4000000000000}, // [1 3 7 15 30 60 121 242]
	/* idx: 486 */ {0x100000004000808a, 0x200000000000000, 0x0, 0x8000000000000}, // [1 3 7 15 30 60 121 243]
	/* idx: 487 */ {0x100000004000808a, 0x200000000000000, 0x0, 0x8000000000000}, // [1 3 7 15 30 60 121 243]
	/* idx: 488 */ {0x200000004000808a, 0x400000000000000, 0x0, 0x10000000000000}, // [1 3 7 15 30 61 122 244]
	/* idx: 489 */ {0x200000004000808a, 0x400000000000000, 0x0, 0x10000000000000}, // [1 3 7 15 30 61 122 244]
	/* idx: 490 */ {0x200000004000808a, 0x400000000000000, 0x0, 0x20000000000000}, // [1 3 7 15 30 61 122 245]
	/* idx: 491 */ {0x200000004000808a, 0x400000000000000, 0x0, 0x20000000000000}, // [1 3 7 15 30 61 122 245]
	/* idx: 492 */ {0x200000004000808a, 0x800000000000000, 0x0, 0x40000000000000}, // [1 3 7 15 30 61 123 246]
	/* idx: 493 */ {0x200000004000808a, 0x800000000000000, 0x0, 0x40000000000000}, // [1 3 7 15 30 61 123 246]
	/* idx: 494 */ {0x200000004000808a, 0x800000000000000, 0x0, 0x80000000000000}, // [1 3 7 15 30 61 123 247]
	/* idx: 495 */ {0x200000004000808a, 0x800000000000000, 0x0, 0x80000000000000}, // [1 3 7 15 30 61 123 247]
	/* idx: 496 */ {0x400000008000808a, 0x1000000000000000, 0x0, 0x100000000000000}, // [1 3 7 15 31 62 124 248]
	/* idx: 497 */ {0x400000008000808a, 0x1000000000000000, 0x0, 0x100000000000000}, // [1 3 7 15 31 62 124 248]
	/* idx: 498 */ {0x400000008000808a, 0x1000000000000000, 0x0, 0x200000000000000}, // [1 3 7 15 31 62 124 249]
	/* idx: 499 */ {0x400000008000808a, 0x1000000000000000, 0x0, 0x200000000000000}, // [1 3 7 15 31 62 124 249]
	/* idx: 500 */ {0x400000008000808a, 0x2000000000000000, 0x0, 0x400000000000000}, // [1 3 7 15 31 62 125 250]
	/* idx: 501 */ {0x400000008000808a, 0x2000000000000000, 0x0, 0x400000000000000}, // [1 3 7 15 31 62 125 250]
	/* idx: 502 */ {0x400000008000808a, 0x2000000000000000, 0x0, 0x800000000000000}, // [1 3 7 15 31 62 125 251]
	/* idx: 503 */ {0x400000008000808a, 0x2000000000000000, 0x0, 0x800000000000000}, // [1 3 7 15 31 62 125 251]
	/* idx: 504 */ {0x800000008000808a, 0x4000000000000000, 0x0, 0x1000000000000000}, // [1 3 7 15 31 63 126 252]
	/* idx: 505 */ {0x800000008000808a, 0x4000000000000000, 0x0, 0x1000000000000000}, // [1 3 7 15 31 63 126 252]
	/* idx: 506 */ {0x800000008000808a, 0x4000000000000000, 0x0, 0x2000000000000000}, // [1 3 7 15 31 63 126 253]
	/* idx: 507 */ {0x800000008000808a, 0x4000000000000000, 0x0, 0x2000000000000000}, // [1 3 7 15 31 63 126 253]
	/* idx: 508 */ {0x800000008000808a, 0x8000000000000000, 0x0, 0x4000000000000000}, // [1 3 7 15 31 63 127 254]
	/* idx: 509 */ {0x800000008000808a, 0x8000000000000000, 0x0, 0x4000000000000000}, // [1 3 7 15 31 63 127 254]
	/* idx: 510 */ {0x800000008000808a, 0x8000000000000000, 0x0, 0x8000000000000000}, // [1 3 7 15 31 63 127 255]
	/* idx: 511 */ {0x800000008000808a, 0x8000000000000000, 0x0, 0x8000000000000000}, // [1 3 7 15 31 63 127 255]
}
